<?php

namespace App\Services\Billing;

use App\Contracts\Billing\BillingProvider;
use App\Models\Plan;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use InvalidArgumentException;

class PaystackBillingProvider implements BillingProvider
{
    private string $secretKey;
    private string $baseUrl;

    public function __construct(private array $config = [])
    {
        $this->secretKey = Arr::get($config, 'secret_key', env('PAYSTACK_SECRET'));
        $this->baseUrl = rtrim(Arr::get($config, 'base_url', 'https://api.paystack.co'), '/');

        if (!$this->secretKey) {
            throw new InvalidArgumentException('Paystack secret key is not configured.');
        }
    }

    public function createCustomer(array $payload): array
    {
        $response = $this->post('/customer', [
            'email' => $payload['email'] ?? Str::uuid()->toString().'@example.com',
            'first_name' => Arr::get($payload, 'first_name'),
            'last_name' => Arr::get($payload, 'last_name'),
            'phone' => Arr::get($payload, 'phone'),
            'metadata' => Arr::get($payload, 'metadata', []),
        ]);

        return $response['data'] ?? [];
    }

    public function createSubscription(array $customer, Plan $plan, array $options = []): array
    {
        $response = $this->post('/subscription', [
            'customer' => $customer['customer_code'] ?? $customer['id'] ?? null,
            'plan' => $plan->code,
            'start_date' => Arr::get($options, 'start_date'),
            'metadata' => Arr::get($options, 'metadata', []),
        ]);

        return $response['data'] ?? [];
    }

    public function swapSubscription(array $subscription, Plan $plan, array $options = []): array
    {
        $subscriptionCode = $subscription['id'] ?? $subscription['subscription_code'] ?? null;

        if (!$subscriptionCode) {
            throw new InvalidArgumentException('Paystack subscription code missing.');
        }

        $response = $this->post('/subscription/enable', [
            'code' => $subscriptionCode,
            'token' => Arr::get($options, 'token'),
        ]);

        // Paystack does not support direct plan swap via API; simulate updated payload
        return array_merge($subscription, [
            'plan_code' => $plan->code,
            'status' => Arr::get($response, 'data.status', 'active'),
        ]);
    }

    public function cancelSubscription(array $subscription, bool $immediate = false): void
    {
        $subscriptionCode = $subscription['id'] ?? $subscription['subscription_code'] ?? null;

        if (!$subscriptionCode) {
            return;
        }

        $this->post('/subscription/disable', [
            'code' => $subscriptionCode,
            'token' => $subscription['email_token'] ?? Arr::get($subscription, 'token'),
        ]);
    }

    public function createPaymentMethod(array $payload): array
    {
        if ($authorization = Arr::get($payload, 'authorization_code')) {
            return [
                'reference' => $authorization,
                'brand' => Arr::get($payload, 'card_brand'),
                'last_four' => Arr::get($payload, 'card_last_four'),
                'exp_month' => Arr::get($payload, 'card_exp_month'),
                'exp_year' => Arr::get($payload, 'card_exp_year'),
                'meta' => [
                    'authorization_code' => $authorization,
                    'signature' => Arr::get($payload, 'signature'),
                    'card' => Arr::only($payload, [
                        'card_brand',
                        'card_last_four',
                        'card_exp_month',
                        'card_exp_year',
                    ]),
                ],
            ];
        }

        $response = $this->post('/customer', [
            'email' => $payload['email'],
            'metadata' => array_merge([
                'payment_method' => true,
            ], Arr::get($payload, 'metadata', [])),
        ]);

        $data = Arr::get($response, 'data', []);

        return [
            'reference' => Arr::get($data, 'customer_code'),
            'brand' => Arr::get($data, 'authorization.brand'),
            'last_four' => Arr::get($data, 'authorization.last4'),
            'exp_month' => Arr::get($data, 'authorization.exp_month'),
            'exp_year' => Arr::get($data, 'authorization.exp_year'),
            'meta' => array_merge($data, [
                'authorization_code' => Arr::get($data, 'authorization.authorization_code'),
            ]),
        ];
    }

    public function attachPaymentMethod(array $customer, array $method): void
    {
        // Paystack payment authorization handled client-side; nothing to do server-side here.
    }

    public function previewPlanChange(array $customer, Plan $plan, array $options = []): array
    {
        return [
            'plan_code' => $plan->code,
            'amount_cents' => $plan->price_cents,
            'currency' => $plan->currency,
            'proration_cents' => 0,
            'notes' => 'Paystack does not provide API proration preview; returning static pricing.',
        ];
    }

    private function post(string $endpoint, array $payload): array
    {
        $response = Http::withToken($this->secretKey)
            ->acceptJson()
            ->post($this->baseUrl.$endpoint, $payload);

        if ($response->failed()) {
            throw new InvalidArgumentException('Paystack API error: '.$response->body());
        }

        return $response->json();
    }
}
