<?php

namespace App\Integrations\Notification;

use App\Integrations\Notification\Contracts\EmailProvider;
use App\Integrations\Notification\Contracts\SmsProvider;
use Illuminate\Contracts\Container\Container;
use Illuminate\Support\Arr;
use InvalidArgumentException;

class NotificationGateway
{
    private array $smsCache = [];
    private array $emailCache = [];

    public function __construct(private Container $container)
    {
    }

    public function sendSms(string|array $recipients, string $message, array $options = [], ?string $driver = null): void
    {
        $provider = $this->resolveSmsProvider($driver);
        $provider->send($recipients, $message, $options);
    }

    public function sendEmail(string $to, string $subject, string $view, array $data = [], array $options = [], ?string $driver = null): void
    {
        $provider = $this->resolveEmailProvider($driver);
        $provider->send($to, $subject, $view, $data, $options);
    }

    private function resolveSmsProvider(?string $driver = null): SmsProvider
    {
        $config = config('notification.sms');
        $driver = $driver ?? Arr::get($config, 'default');

        if (!$driver) {
            throw new InvalidArgumentException('Notification SMS driver is not configured.');
        }

        if (isset($this->smsCache[$driver])) {
            return $this->smsCache[$driver];
        }

        $driverConfig = Arr::get($config, "drivers.{$driver}");

        if (!$driverConfig || !isset($driverConfig['class'])) {
            throw new InvalidArgumentException("SMS driver [{$driver}] is not configured.");
        }

        $class = $driverConfig['class'];
        $payload = Arr::except($driverConfig, ['class']);

        return $this->smsCache[$driver] = $this->container->make($class, ['config' => $payload]);
    }

    private function resolveEmailProvider(?string $driver = null): EmailProvider
    {
        $config = config('notification.email');
        $driver = $driver ?? Arr::get($config, 'default');

        if (!$driver) {
            throw new InvalidArgumentException('Notification email driver is not configured.');
        }

        if (isset($this->emailCache[$driver])) {
            return $this->emailCache[$driver];
        }

        $driverConfig = Arr::get($config, "drivers.{$driver}");

        if (!$driverConfig || !isset($driverConfig['class'])) {
            throw new InvalidArgumentException("Email driver [{$driver}] is not configured.");
        }

        $class = $driverConfig['class'];
        $payload = Arr::except($driverConfig, ['class']);

        return $this->emailCache[$driver] = $this->container->make($class, ['config' => $payload]);
    }
}
