<?php

namespace App\Http\Controllers;

use App\Models\Family;
use App\Models\Invitation;
use App\Services\ActivityLogger;
use App\Services\InvitationService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class InvitationController extends Controller
{
    private InvitationService $invitations;
    private ActivityLogger $activityLogger;

    public function __construct(InvitationService $invitations, ActivityLogger $activityLogger)
    {
        $this->invitations = $invitations;
        $this->activityLogger = $activityLogger;
    }

    public function index(Request $request, Family $family)
    {
        $this->authorizeManage($request);

        $perPage = (int) $request->integer('per_page', 15);
        $perPage = $perPage > 0 ? min($perPage, 100) : 15;

        $invitations = $this->invitations->listForFamily($family, $perPage);

        return response()->json($invitations);
    }

    public function store(Request $request, Family $family)
    {
        $this->authorizeManage($request);

        $data = $request->validate([
            'email' => ['nullable', 'email', 'required_without:phone'],
            'phone' => ['nullable', 'string', 'required_without:email'],
            'token' => ['nullable', 'string'],
            'expires_at' => ['nullable', 'date', 'after:now'],
            'accepted_person_id' => ['nullable', 'integer', 'exists:people,id'],
        ]);

        $invitation = $this->invitations->createInvitation($family, $data, $request->user());

        $this->activityLogger->log('invitation.created', $invitation, [
            'family_id' => $family->id,
        ], $request, $family->id);

        return response()->json($invitation, 201);
    }

    public function destroy(Request $request, Family $family, Invitation $invitation)
    {
        $this->authorizeManage($request);

        abort_unless($invitation->family_id === $family->id, 404);

        $this->invitations->revokeInvitation($invitation);

        $this->activityLogger->log('invitation.revoked', $invitation, [
            'family_id' => $family->id,
        ], $request, $family->id);

        return response()->json(['message' => 'Invitation revoked']);
    }

    public function accept(Request $request)
    {
        $data = $request->validate([
            'token' => ['required', 'string'],
            'accepted_person_id' => ['nullable', 'integer', 'exists:people,id'],
        ]);

        $invitation = Invitation::where('token', $data['token'])->firstOrFail();

        if ($invitation->expires_at instanceof Carbon && $invitation->expires_at->isPast()) {
            abort(410, 'Invitation has expired.');
        }

        if ($invitation->status !== 'pending') {
            abort(409, 'Invitation has already been processed.');
        }

        $updated = $this->invitations->acceptInvitation($invitation, $data['accepted_person_id'] ?? null);

        $this->activityLogger->log('invitation.accepted', $updated, [
            'family_id' => $invitation->family_id,
        ], $request, $invitation->family_id);

        return response()->json($updated);
    }

    private function authorizeManage(Request $request): void
    {
        abort_unless($request->user()?->can('invitations.manage'), 403);
    }
}
