<?php

namespace App\Http\Controllers;

use App\Models\Family;
use App\Models\FamilyAnnouncement;
use App\Models\Person;
use App\Notifications\FamilyAnnouncementNotification;
use App\Services\NotificationRelayService;
use App\Integrations\Notification\NotificationGateway;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FamilyAnnouncementController extends Controller
{
    public function index(Request $request, Family $family)
    {
        $this->authorizeMember($request, $family);

        $announcements = FamilyAnnouncement::query()
            ->where('family_id', $family->id)
            ->with('recipients')
            ->latest('created_at')
            ->paginate(15);

        $members = $family->members()
            ->select('people.id', 'first_name', 'last_name')
            ->orderBy('first_name')
            ->get()
            ->map(fn($p) => [
                'id' => $p->id,
                'name' => trim(($p->first_name ?? '') . ' ' . ($p->last_name ?? '')) ?: 'Member',
            ]);

        $canManage = $this->canManage($request, $family);
        $canCreate = $this->canCreate($request, $family);
        $creatorIds = $family->members()->wherePivot('can_post_announcements', true)->pluck('people.id')->map(fn($id) => (int) $id)->all();
        $prefill = null;
        $editId = (int) $request->query('edit', 0);
        if ($editId > 0) {
            $a = FamilyAnnouncement::where('family_id', $family->id)->where('id', $editId)->first();
            if ($a) {
                $prefill = [
                    'id' => $a->id,
                    'title' => $a->title,
                    'content' => $a->content,
                    'send_to_all' => $a->send_to_all,
                    'recipients' => $a->recipients()->pluck('people.id')->all(),
                ];
            }
        }

        return view('families.announcements.index', [
            'family' => $family,
            'announcements' => $announcements,
            'members' => $members,
            'canManage' => $canManage,
            'canCreate' => $canCreate,
            'creatorIds' => $creatorIds,
            'prefill' => $prefill,
        ]);
    }

    public function store(Request $request, Family $family)
    {
        $this->authorizeCreate($request, $family);

        $data = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'content' => ['required', 'string'],
            'send_to_all' => ['nullable', 'boolean'],
            'recipients' => ['array'],
            'recipients.*' => ['integer'],
        ]);

        $sendAll = (bool)($data['send_to_all'] ?? false);

        // Ensure recipients belong to this family when specified
        $allowedIds = $family->members()->pluck('people.id')->map(fn($id) => (int)$id)->all();
        $recipients = collect($data['recipients'] ?? [])
            ->map(fn($id) => (int)$id)
            ->filter(fn($id) => in_array($id, $allowedIds, true))
            ->unique()
            ->values()
            ->all();

        DB::transaction(function () use ($family, $request, $data, $sendAll, $recipients) {
            $announcement = FamilyAnnouncement::create([
                'family_id' => $family->id,
                'created_by' => $request->user()?->id,
                'title' => $data['title'],
                'content' => $data['content'],
                'send_to_all' => $sendAll,
            ]);
            if (!$sendAll && !empty($recipients)) {
                $announcement->recipients()->sync($recipients);
            }

            // Notify recipients
            $relay = app(NotificationRelayService::class);
            $gateway = app(NotificationGateway::class);
            $peopleQuery = $sendAll
                ? $family->members()->with('user')
                : Person::query()->whereIn('id', $recipients)->with('user');
            $peopleQuery->get()->each(function (Person $p) use ($family, $announcement, $relay, $gateway) {
                if ($p->user) {
                    $p->user->notify(new FamilyAnnouncementNotification($family, $announcement));
                }
                $phone = is_array($p->meta ?? null) ? ($p->meta['phone'] ?? null) : null;
                $text = 'New announcement in ' . $family->name . ': ' . $announcement->title;
                try {
                    if ($phone) {
                        $gateway->sendSms($phone, $text);
                    }
                } catch (\Throwable $e) {
                    // ignore sms failures
                }
                $relay->sendWhatsApp($phone, $text);
            });
        });

        return redirect()->route('families.announcements.index', $family)->with('status', 'Announcement sent.');
    }

    private function authorizeMember(Request $request, Family $family): void
    {
        $user = $request->user();
        $isMember = $user && $family->members()->where('people.user_id', $user->id)->exists();
        abort_unless($isMember, 403);
    }

    private function authorizeManage(Request $request, Family $family): void
    {
        $user = $request->user();
        $isOwner = $user && $family->members()
            ->wherePivot('role', 'owner')
            ->where('people.user_id', $user->id)
            ->exists();
        abort_unless($user && ($user->can('families.manage') || $isOwner), 403);
    }

    private function canCreate(Request $request, Family $family): bool
    {
        $user = $request->user();
        if (!$user) return false;
        if ($this->canManage($request, $family)) return true;
        return $family->members()
            ->where('people.user_id', $user->id)
            ->wherePivot('can_post_announcements', true)
            ->exists();
    }

    private function authorizeCreate(Request $request, Family $family): void
    {
        abort_unless($this->canCreate($request, $family), 403);
    }

    private function canManage(Request $request, Family $family): bool
    {
        $user = $request->user();
        $isOwner = $user && $family->members()
            ->wherePivot('role', 'owner')
            ->where('people.user_id', $user->id)
            ->exists();
        return $user && ($user->can('families.manage') || $isOwner);
    }

    public function update(Request $request, Family $family, FamilyAnnouncement $announcement)
    {
        $this->authorizeManage($request, $family);
        abort_unless($announcement->family_id === $family->id, 404);

        $data = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'content' => ['required', 'string'],
            'send_to_all' => ['nullable', 'boolean'],
            'recipients' => ['array'],
            'recipients.*' => ['integer'],
        ]);

        $sendAll = (bool)($data['send_to_all'] ?? false);
        $allowedIds = $family->members()->pluck('people.id')->map(fn($id) => (int)$id)->all();
        $recipients = collect($data['recipients'] ?? [])
            ->map(fn($id) => (int)$id)
            ->filter(fn($id) => in_array($id, $allowedIds, true))
            ->unique()
            ->values()
            ->all();

        DB::transaction(function () use ($announcement, $data, $sendAll, $recipients) {
            $announcement->title = $data['title'];
            $announcement->content = $data['content'];
            $announcement->send_to_all = $sendAll;
            $announcement->save();
            if ($sendAll) {
                $announcement->recipients()->sync([]);
            } else {
                $announcement->recipients()->sync($recipients);
            }
        });

        return redirect()->route('families.announcements.index', $family)->with('status', 'Announcement updated.');
    }

    public function destroy(Request $request, Family $family, FamilyAnnouncement $announcement)
    {
        $this->authorizeManage($request, $family);
        abort_unless($announcement->family_id === $family->id, 404);
        $announcement->delete();
        return redirect()->route('families.announcements.index', $family)->with('status', 'Announcement deleted.');
    }

    public function updatePermissions(Request $request, Family $family)
    {
        $this->authorizeManage($request, $family);
        $data = $request->validate([
            'creators' => ['array'],
            'creators.*' => ['integer'],
        ]);

        $allowedIds = $family->members()->pluck('people.id')->map(fn($id) => (int)$id)->all();
        $selected = collect($data['creators'] ?? [])
            ->map(fn($id) => (int)$id)
            ->filter(fn($id) => in_array($id, $allowedIds, true))
            ->unique()
            ->values()
            ->all();

        foreach ($allowedIds as $pid) {
            $family->members()->updateExistingPivot($pid, [
                'can_post_announcements' => in_array($pid, $selected, true),
            ]);
        }

        return redirect()->route('families.announcements.index', $family)->with('status', 'Announcement permissions updated.');
    }
}
